// File_Ogg - Info for ogg files
// Copyright (C) 2002-2006 Jerome Martinez, Zen@MediaArea.net
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

//---------------------------------------------------------------------------
// Compilation condition
#if defined(MEDIAINFO_OGG_YES) || (!(defined(MEDIAINFO_VIDEO_NO) && defined(MEDIAINFO_AUDIO_NO)) && !defined(MEDIAINFO_OGG_NO))
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
#include <wx/wxprec.h>
#ifdef __BORLANDC__
    #pragma hdrstop
#endif
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
#include <ZenLib/ZtringListList.h>
#include <ZenLib/Utils.h>
#include "MediaInfo/Multiple/File_Ogg.h"
using namespace ZenLib;
//---------------------------------------------------------------------------

namespace MediaInfoLib
{

//***************************************************************************
// Functions
//***************************************************************************

int File_Ogg::Read(const int8u* Begin_, size_t Begin_Size_, const int8u* End_, size_t End_Size_, int64u FileSize)
{
    //Init
    Begin=Begin_;
    Begin_Size=Begin_Size_;
    End=End_;
    End_Size=End_Size_;
    Offset=0;
    ShouldStop=false;

    //Integrity test
    if (Begin_Size<128 || CC4(Begin)!=CC4("OggS"))
        return -1;

    Stream_Prepare(Stream_General);
    Fill("Format", "Ogg");

    while (!ShouldStop)
    {
        ChunkHeader_Analyse();
        if (!ShouldStop)
        {
            ChunkData_Analyse();
        }
    }

    //Some headers my have not comments
    if (!ChunkData_Offset.empty() && ChunkData_Offset.size()==ChunkData_Size.size())
        for (size_t Pos=0; Pos<ChunkData_Offset.size(); Pos++)
            if (Begin[ChunkData_Offset[Pos]]==0x01 || Begin[ChunkData_Offset[Pos]]==0x80) //This is a identifier
                Identification_Analyse(ChunkData_Offset[Pos]+1, ChunkData_Size[Pos]-1);



/*
    int Taille=4; //Taille du bloc en cours
    size_t Offset=0; //Offset debut

    char Commentaire[10]; //Type (A = Audio, V = Video)
    int Commentaire_I[10]; //offset du type (Audio, video...)
    int Commentaire_Nb[10]; //offset du flux
    int Commentaire_Pos=-1; //nombre de commentaires a prevoir
    int Commentaire_EnCours=-1; //position du commentaire en cours

    while (Offset+Taille<Begin_Size-100)
    {
        Taille=4;
        while (Offset+Taille<Begin_Size-4 && (Begin[Offset+Taille]!='O' || Begin[Offset+Taille+1]!='g' || Begin[Offset+Taille+2]!='g' || Begin[Offset+Taille+3]!='S'))
            Taille++;
        if (Offset+Taille<Begin_Size)
        {
            int Debut=Offset+28+Begin[Offset+26]; //Debut de la trame reelle (apres OggS et le type)
            int Id=Begin[Debut-1];
            if (Id==1)
            {
                //c'est une info - Offset a remplacer par Debut
                Commentaire_Pos++;
                Commentaire_Nb[Commentaire_Pos]=(int)Begin[Offset+14]; //Numero de flux
                //Recuperation codec si pas vorbis
                char C1[5]; C1[4]='\0';
                Ztring Format; //Celui de StreamFormat, le vrai format
                Format.From_Local((char*)Begin+Offset+37, 4);

                if (Begin[Offset+29]=='v' && Begin[Offset+30]=='i') //C'est une video
                {
                    size_t Video_Count=Stream_Prepare(Stream_Video);
                    Video[Video_Count](_T("Codec"))=Format;
                    Video[Video_Count](_T("FrameRate")).From_Number((float)10000000/(float)LittleEndian2int32u((char*)Begin+Offset+45), 3);
                    //time normalement de 53 a 60 mais a 1 (8 octets)
                    //time normalement de 61 a 64 mais a 1 (4 octets)
                    //buffer de 65 a 68
                    //bits (24 bits) de 69 a 72
                    Video[Video_Count](_T("Width")).From_Number(LittleEndian2int32u((char*)Begin+Offset+73));
                    Video[Video_Count](_T("Height")).From_Number(LittleEndian2int32u((char*)Begin+Offset+77));
                    Commentaire[Commentaire_Pos]='V'; Commentaire_I[Commentaire_Pos]=Video_Count;
                }
                if (Begin[Offset+29]=='v' && Begin[Offset+30]=='o') //C'est un son Vorbis
                {
                    size_t Audio_Count=Stream_Prepare(Stream_Audio);
                    Audio[Audio_Count](_T("Codec"))=_T("Vorbis");
                    Audio[Audio_Count](_T("BitRate")).From_Number((int)LittleEndian2int64s((char*)Begin+Offset+48)); if (Audio[Audio_Count](_T("BitRate"))==_T("-8")) Audio[Audio_Count](_T("BitRate"))=_T(""); //BitRate de 48 a 63
                    Audio[Audio_Count](_T("Channel(s)")).From_Number(LittleEndian2int8u((char*)Begin+Offset+39)); //Canaux a 39
                    Audio[Audio_Count](_T("SamplingRate")).From_Number((int)LittleEndian2int64u((char*)Begin+Offset+40)); //Frequence  de 40 a 47
                    Commentaire[Commentaire_Pos]='A'; Commentaire_I[Commentaire_Pos]=Audio_Count;
                }
                if (Begin[Offset+29]=='a' && Begin[Offset+30]=='u') //C'est un son Audio
                {
                    size_t Audio_Count=Stream_Prepare(Stream_Audio);
                    Audio[Audio_Count](_T("Codec"))=Format;
                    Audio[Audio_Count](_T("BitRate")).From_Number(LittleEndian2int32s((char*)Begin+Offset+77)*8); if (Audio[Audio_Count](_T("BitRate"))==_T("-8")) Audio[Audio_Count](_T("BitRate"))=_T("");
                    Audio[Audio_Count](_T("Channel(s)")).From_Number(LittleEndian2int8u((char*)Begin+Offset+73));
                    Audio[Audio_Count](_T("SamplingRate")).From_Number((int)LittleEndian2int32u((char*)Begin+Offset+53));
                    Commentaire[Commentaire_Pos]='A'; Commentaire_I[Commentaire_Pos]=Audio_Count;
                }
                if (Begin[Offset+29]=='t' && Begin[Offset+30]=='e') //C'est du texte
                {
                    size_t Text_Count=Stream_Prepare(Stream_Text);
                    Text[Text_Count](_T("Codec"))=_T("UTF-8");
                    Commentaire[Commentaire_Pos]='T'; Commentaire_I[Commentaire_Pos]=Text_Count;
                }
            }
            else if (Id==3)
            {
                //c'est un commentaire
                Commentaire_EnCours=0;
                while (Commentaire_Nb[Commentaire_EnCours]!=(int)Begin[Offset+14] && Commentaire_EnCours<10)
                    Commentaire_EnCours++;
                //vendorlenght, vendor, Count de commentaire, commentair lenght, commentaire
                unsigned int Pos=Debut+6;
                int *I1=(int*)(Begin+Pos);
                Pos+=I1[0]+4;
                if (Pos>Begin_Size)
                    return 1;   //TODO : see why this problem
                I1=(int*)(Begin+Pos);
                int Count=I1[0];
                Pos+=4;//Place au premiere taille de commentaire
                for (int I2=0; I2<Count; I2++) // Pour chaque commentaire
                {
                    //recuperation d'un commentaire
                    I1=(int*)(Begin+Pos);
                    if (I1[0]>65536-1) //TODO : see why this problem //To prevent crash if this is a corrupted file
                        I1[0]=65536-1;
                    Ztring C2;
                    C2.From_UTF8((char*)Begin+Pos+4, I1[0]);
                    #ifndef UNICODE //if not Unicode, must translate in locale
                        int CharSize=wxConvCurrent->MB2WC(NULL, C2.c_str(), 0);
                        char* MediaInfo_Char=new char[CharSize+1]; MediaInfo_Char[CharSize]=0;
                        wchar_t* MediaInfo_WChar=new wchar_t[CharSize+1];
                        wxConvUTF8.MB2WC(MediaInfo_WChar, C2.c_str(), CharSize); //into WideChar
                        wxConvCurrent->WC2MB(MediaInfo_Char, MediaInfo_WChar, CharSize); //into WideChar
                        C2=MediaInfo_Char;
                        delete[] MediaInfo_Char;
                        delete[] MediaInfo_WChar;
                    #endif //UNICODE
                    Ztring ID=C2.SubString(_T(""), _T("="));
                    Ztring Texte;
                    if (ID!=_T(""))
                        Texte=C2.SubString(_T("="), _T(""));
                    else //Out of specifications!!! Grrr Doom9 :)
                        Texte=C2;

                    //Gestion
                         if (ID==_T("TITLE")) General[0](_T("Title"))=Texte;
                    else if (ID==_T("VERSION")) General[0](_T("Title/More"))=Texte;
                    else if (ID==_T("ARTIST")) General[0](_T("Author"))=Texte;
                    else if (ID==_T("LOCATION")) General[0](_T("RecordingLocation"))=Texte;
                    else if (ID==_T("ALBUM")) General[0](_T("Album"))=Texte;
                    else if (ID==_T("DATE")) General[0](_T("Date"))=Texte;
                    else if (ID==_T("COMMENTS")) General[0](_T("Comment"))=Texte;
                    else if (ID==_T("TRACKNUMBER")) General[0](_T("Track"))=Texte;
                    else if (ID==_T("AUTHOR")) General[0](_T("Author"))=Texte;
                    else if (ID==_T("COPYRIGHT")) General[0](_T("Copyright"))=Texte;
                    else if (ID==_T("DESCRIPTION")) General[0](_T("Comment"))=Texte;
                    else if (ID.find(_T("CHAPTER"))!=(size_t)-1)
                    {
                        if (Chapters.empty())
                            Stream_Prepare(Stream_Chapters);
                        if (ID.find(_T("NAME"))==(size_t)-1)
                        {
                            Chapters[0](_T("Total"))=ID.SubString(_T("CHAPTER"), _T(""));
                            int32s Chapter=Chapters[0](_T("Total")).To_int32s();
                            Chapters[0](10+Chapter, 0)=Chapters[0](_T("Total"));
                            Chapters[0](10+Chapter, Info_Name_Text)=Chapters[0](_T("Total"));
                            Chapters[0](10+Chapter, 1)=C2.SubString(_T("="), _T(""));

                        }
                        else
                        {
                            Chapters[0](10+Chapters[0](_T("Total")).To_int32s(), 1)+=Ztring(_T(" "))+C2.SubString(_T("="), _T(""));
                        }
                    }
                    else if (ID==_T("LANGUAGE") && Commentaire[Commentaire_EnCours]=='A') {Audio[Commentaire_I[Commentaire_EnCours]](_T("Language/String"))=Texte; Audio[Commentaire_I[Commentaire_EnCours]](_T("Language"))=Ztring(Texte.c_str(),2);}
                    else if (ID==_T("LANGUAGE") && Commentaire[Commentaire_EnCours]=='T') {Text[Commentaire_I[Commentaire_EnCours]](_T("Language/String"))=Texte; Text[Commentaire_I[Commentaire_EnCours]](_T("Language"))=Ztring(Texte.c_str(),2);}
                    else if (Commentaire[Commentaire_EnCours]=='A')
                    {
                        //TODO : Chaine_Tableau2 ou 1 gere mal quand Chaine[0] est vide et pas Chaine[1]
                        //TODO : RechercherEntre : si pas _T("="), retorune Begin, pas cool
                        if (ID==_T(""))
                        {
                            ZtringList C3=Ztring(Ztring(_T("(Note by MediaInfo, Out of specifications);"))+Texte);
                            C3(Info_Name_Text)=_T("(Note by MediaInfo, Out of specifications)");
                            Audio[Commentaire_I[Commentaire_EnCours]].push_back(C3);
                        }
                        else
                        {
                            ZtringList C3=Ztring(C2.SubString(_T(""), _T("="))+_T(";")+C2.SubString(_T("="), _T("")));
                            Ztring C4=C3(0);
                            C3(Info_Name_Text)=C4;
                            Audio[Commentaire_I[Commentaire_EnCours]].push_back(C3);
                        }
                    }
                    else if (Commentaire[Commentaire_EnCours]=='T') Text[Commentaire_I[Commentaire_EnCours]].push_back((C2.SubString(_T(""), _T("="))+_T(";")+C2.SubString(_T("="), _T(""))).c_str()); //Visual C++ patch, doesn't like push_back overload
                    else
                    {
                         General[0].push_back(C2.SubString(_T(""), _T("=")));
                         General[0](C2.SubString(_T(""), _T("=")), Info_Name_Text)=C2.SubString(_T(""), _T("="));
                         General[0](C2.SubString(_T(""), _T("=")))=C2.SubString(_T("="), _T(""));
                    }

                    //On positionne correctement
                    Pos+=I1[0]+4;
                    if (Pos>65536)  //To prevent crash if this is a corrupted file
                        break;
                }
            }
            Offset+=Taille;
            Taille=0;
        }
    }

    Offset=End_Size-100;
    while (Offset>0 && (End[Offset]!='O' || End[Offset+1]!='g' || End[Offset+2]!='g' || End[Offset+3]!='S' || End[Offset+14]!=Commentaire_Nb[0]))
        Offset--;
    if (Offset>=0)
    {
        //Recherche duree
        float F1=(float)LittleEndian2int32u((char*)End+Offset+6); //Granule Pos
        Commentaire_EnCours=0;
        while (Commentaire_Nb[Commentaire_EnCours]!=(int)End[Offset+14] && Commentaire_EnCours<10)
            Commentaire_EnCours++;
        if (Commentaire[Commentaire_EnCours]=='V')
            if (Video[Commentaire_I[Commentaire_EnCours]](_T("FrameRate")).To_float32()!=0)
                General[0](_T("PlayTime")).From_Number((int)(F1*1000/Video[Commentaire_I[Commentaire_EnCours]](_T("FrameRate")).To_float32()));
        else
            if (Audio[Commentaire_I[Commentaire_EnCours]](_T("SamplingRate")).To_float32()!=0)
                General[0](_T("PlayTime")).From_Number((int)(F1*1000/Audio[Commentaire_I[Commentaire_EnCours]](_T("SamplingRate")).To_float32()));

        if (General[0](_T("PlayTime")).To_int32s()>1000) //Pas pratique si <1s!!!
        {
            //Overal BitRate
            General[0](_T("OveralBitRate")).From_Number((int)(8*(float)General[0](_T("FileSize")).To_int32s()/(General[0](_T("PlayTime")).To_int32s()/1000)));
            //Can we calculate video bitrate?
            bool VideoBitRate_Calculate=true;
            for (size_t I1=0; I1<Audio.size(); I1++)
                if (Audio[I1](_T("BitRate")).size()==0 || Audio[I1](_T("BitRate"))==_T("-1"))
                {
                    VideoBitRate_Calculate=false; //At least 1 audio without bitrate, can not calculate video bitrate
                    Audio[I1](_T("BitRate")).clear();//No need to keep "-1"
                }
            //Video bitrate
            if (VideoBitRate_Calculate)
            {
                int32u VideoBitRate=(int32u)(General[0](_T("OveralBitRate")).To_int32u()*0.9897); //0.9897 car 1.03% d'overhead!!!
                for (size_t I1=0; I1<Audio.size(); I1++)
                    VideoBitRate-=Audio[I1](_T("BitRate")).To_int32u();
                Video[0](_T("BitRate")).From_Number(VideoBitRate);
            }
        }
    }

    //Gestion cas speciaux LANGUAGE
    for (size_t Pos=0; Pos<Audio.size(); Pos++)
    {
        if (Audio[Pos](_T("Language"))==_T("Di"))
        {
            Audio[Pos](_T("Language"))=_T("En");
            Audio[Pos](_T("Language/String"))=_T("English");
            Audio[Pos](_T("Language/Info"))=_T("Director's comments");
        }
    }


*/
    return 1;
}

//---------------------------------------------------------------------------
// ChunkHeader
// "OggS"                           4 bytes, Pos=0
// Version                          1 byte, Pos=4
// Flags                            1 byte, Pos=5
// absolute granule position        8 bytes, Pos=6
// SreamID                          4 byte, Pos=14
// Page counter                     4 bytes, Pos=18
// Checksum                         4 bytes, Pos=22
// DataSizeSize                     1 bytes, Pos=26
// DataSize                         DataSizeSize bytes, Pos=27, Xiph method
//
// Flags:
// Fresh/Continued packet
// Not first/first page of logical bitstream (BOS)
// Not last/last page of logical bitstream (EOS)
//
void File_Ogg::ChunkHeader_Analyse()
{
    //Look for header
    while (Offset+28<Begin_Size && CC4(Begin)!=CC4("OggS"))
        Offset++;
    if (Offset+28>=Begin_Size)
    {
        ShouldStop=true;
        return;
    }

    //DataSizeSize
    int8u DataSizeSize=Begin[Offset+26];
    if (Offset+26+DataSizeSize>=Begin_Size)
    {
        ShouldStop=true;
        return;
    }

    //DataSize
    Size=0;
    for (int8u Pos=0; Pos<DataSizeSize; Pos++)
        Size+=Begin[Offset+27+Pos];
    if (Offset+27+DataSizeSize+Size>=Begin_Size)
    {
        ShouldStop=true;
        return;
    }

    //Save position and ID
    size_t Pos=ChunkHeader_Offset.size();
    ChunkHeader_Offset.resize(Pos+1);
    ChunkHeader_ID.resize(Pos+1);
    ChunkData_Offset.resize(Pos+1);
    ChunkData_Size.resize(Pos+1);
    ChunkHeader_Offset[Pos]=Offset;
    ChunkHeader_ID[Pos]=LittleEndian2int32u(Begin+Offset+14);
    ChunkData_Offset[Pos]=Offset+27+DataSizeSize;
    ChunkData_Size[Pos]=Size;

    //Set Offset
    Offset+=27+DataSizeSize;
}

//---------------------------------------------------------------------------
// ChunkData
// If First byte&0x01==1, this is a header packet:
// Type                             1 bytes, Pos=0
// Datas                            X bytes, Pos=1
//
// If First byte&0x01==0, this is a data packet:
// 0                                1 bit (Indicates data packet)
// Bit 2 of lenbytes                1 bit
// unused                           1 bit
// Keyframe                         1 bit
// unused                           2 bits
// Bit 0 of lenbytes                1 bit
// Bit 1 of lenbytes                1 bit
// SamplesCount                     lenbytes bytes (Frames for video, Samples for audio, 1ms units for text)
// Data                             X bytes, Pos=1+lenbytes
void File_Ogg::ChunkData_Analyse()
{
    //Integrity
    if (Size<1)
        return;

    //Parse
         if (0) ;
    else if (Begin[Offset]==0x01) Identification_Analyse();
    else if (Begin[Offset]==0x80) Identification_Analyse();
    else if (Begin[Offset]==0x03) Comment_Analyse();
    else if (Begin[Offset]==0x81) Comment_Analyse();
    else if (Begin[Offset]==0x05) Setup_Analyse();
    else if (Begin[Offset]&0x81==0)
    {
    }

    Offset+=Size;
}

//---------------------------------------------------------------------------
// Stream chunk
// StreamKind                       6-8 bytes, Pos=1
// ... (Depend of StreamKind)
//
void File_Ogg::Identification_Analyse()
{
    //We do nothing here, only save offset
}

//---------------------------------------------------------------------------
// Stream chunk
// StreamKind                       6-8 bytes, Pos=1
// ... (Depend of StreamKind)
//
void File_Ogg::Identification_Analyse(int32u ID)
{
    //We should have comment chunk, we can start to analyse, starting to find the corresponding Identification chunk

    if (ChunkHeader_ID.empty() || ChunkHeader_ID.size()!=ChunkHeader_Offset.size() || ChunkHeader_ID.size()!=ChunkData_Offset.size() || ChunkHeader_ID.size()!=ChunkData_Size.size())
        return;

    size_t Pos=0;
    while (ChunkHeader_ID[Pos]!=ID)
        Pos++;
    if (Pos==ChunkHeader_ID.size()-1)
        return;
    Identification_Analyse(ChunkData_Offset[Pos]+1, ChunkData_Size[Pos]-1);

    //Erase Header chunk
    ChunkHeader_ID.erase(ChunkHeader_ID.begin()+Pos);
    ChunkHeader_Offset.erase(ChunkHeader_Offset.begin()+Pos);
    ChunkData_Offset.erase(ChunkData_Offset.begin()+Pos);
    ChunkData_Size.erase(ChunkData_Size.begin()+Pos);

    //Erase Comment chunk
    ChunkHeader_ID.erase(ChunkHeader_ID.end()-1);
    ChunkHeader_Offset.erase(ChunkHeader_Offset.end()-1);
    ChunkData_Offset.erase(ChunkData_Offset.end()-1);
    ChunkData_Size.erase(ChunkData_Size.end()-1);
}

//---------------------------------------------------------------------------
// Identification chunk, video
// StreamKind                       8 bytes, Pos=0
// Datas                            X bytes, Pos=4
//
void File_Ogg::Identification_Analyse(size_t Offset, size_t Size)
{
    //Integrity
    if (Size<8)
        return;

    if (0) ;
    else if (CC6(Begin+Offset)==CC6("vorbis"))       Identification_Vorbis_Analyse(Offset, Size);
    else if (CC6(Begin+Offset)==CC6("theora"))       Identification_Theora_Analyse(Offset, Size);
    else if (CC8(Begin+Offset)==CC8("video\0\0\0"))  Identification_Video_Analyse (Offset, Size);
    else if (CC8(Begin+Offset)==CC8("audio\0\0\0"))  Identification_Audio_Analyse (Offset, Size);
    else if (CC8(Begin+Offset)==CC8("text\0\0\0\0")) Identification_Text_Analyse  (Offset, Size);
}

//---------------------------------------------------------------------------
// Identification chunk, Vorbis, at least 29 bytes
// "vorbis"                         6 bytes, Pos=0
// Version                          4 bytes, Pos=6
// Channels                         1 bytes, Pos=10
// Samplingrate                     4 bytes, Pos=11
// BitRate_Maximum                  4 bytes, Pos=15
// BitRate_Nominal                  4 bytes, Pos=19
// BitRate_Minimum                  4 bytes, Pos=23
// BlockSize_0                      4 bits, Pos=27 (2^Value)
// BlockSize_1                      4 bits, Pos=27.4 (2^Value)
// Framing                          1 bits, Pos=28
//
 void File_Ogg::Identification_Vorbis_Analyse(size_t Offset, size_t Size)
{
    //Integrity
    if (Size<29)
        return;

    Stream_Prepare(Stream_Audio);
    Fill("Codec", "Vorbis");
    if (LittleEndian2int32u(Begin+Offset+15)<0x80000000) //This is a signed value, and negative values are not OK
        Fill("BitRate", LittleEndian2int32u(Begin+Offset+19)); //BitRate_Nominal
    Fill("Channel(s)", LittleEndian2int8u(Begin+Offset+10)); //Channess
    Fill("SamplingRate", LittleEndian2int32u(Begin+Offset+11));
}

//---------------------------------------------------------------------------
// Identification chunk, Theora, at least 41 bytes
// "theora"                         6 bytes, Pos=0
// Version                          3 bytes, Pos=6
// Width_Blocks                     2 bytes, Pos=9
// Height_Blocks                    2 bytes, Pos=11
// Width                            3 bytes, Pos=13 (displayable width)
// Height                           3 bytes, Pos=16 (displayable heigth)
// Pos_X                            1 byte, Pos=19 (Pixels between 0 and Pox_X are not displayable)
// Pos_Y                            1 byte, Pos=20 (Pixels between 0 and Pox_Y are not displayable)
// FrameRate_N                      4 bytes, Pos=21
// FrameRate_D                      4 bytes, Pos=25 (FrameRate=FrameRate_N/FrameRate_D)
// PixelRatio_N                     3 bytes, Pos=29
// PixelRatio_D                     3 bytes, Pos=32 (PixelRatio=PixelRatio_N/PixelRatio_D, may be zero)
// ColorSpace                       1 byte, Pos=35
// NOMBR (VBR???)                   3 bytes, Pos=36
// Quality                          6 bits, Pos=39 (depend of encoder)
// KFGSHIFT???                      5 bits, Pos=
// Pixel_Format                     2 bits, Pos= (0=4:2:0, 2=4:2:2, 3=4:4:4)
// Reserved                         3 bits, Pos=
//
 void File_Ogg::Identification_Theora_Analyse(size_t Offset, size_t Size)
{
    //Integrity
    if (Size<41)
        return;

    Stream_Prepare(Stream_Video);
    Fill("Codec", "Theora");
    if (BigEndian2int16u(Begin+Offset+6)>0x0302) //Version 3.2.x
        return;
    Fill("FrameRate", ((float)BigEndian2int32u(Begin+Offset+21))/(float)BigEndian2int32u(Begin+Offset+25), 3);
    int32u Width=BigEndian2int32u(Begin+Offset+12)&0x00FFFFFF; //Only 24 bits
    int32u Height=BigEndian2int32u(Begin+Offset+15)&0x00FFFFFF; //Only 24 bits
    int32u PixelRatio_N=BigEndian2int32u(Begin+Offset+29);
    int32u PixelRatio_D=BigEndian2int32u(Begin+Offset+32);
    float PixelRatio=1;
    if (PixelRatio_N && PixelRatio_D)
        PixelRatio=((float)PixelRatio_N)/(float)PixelRatio_D;
    Fill("Width", Width);
    Fill("Height", Height);
    Fill("AspectRatio", ((float)Width)/((float)Height)*PixelRatio, 3);
}

//---------------------------------------------------------------------------
// Identification chunk, video, at least 52 bytes
// "video\0\0\0"                    8 bytes, Pos=0
// fccHandler                       4 bytes, Pos=8
// SizeOfStructure                  4 bytes, Pos=12
// TimeUnit                         8 bytes, Pos=16 (10000000/TimeUnit is stream tick rate in ticks/sec)
// SamplesPerUnit                   8 bytes, Pos=24
// DefaultLengh                     4 bytes, Pos=32 (in media time)
// BufferSize                       4 bytes, Pos=36
// BitsPerSample                    2 bytes, Pos=40
// Reserved                         2 bytes, Pos=42
// Width                            4 bytes, Pos=44
// Height                           4 bytes, Pos=48
//
void File_Ogg::Identification_Video_Analyse(size_t Offset, size_t Size)
{
    //Integrity
    if (Size<52)
        return;

    Stream_Prepare(Stream_Video);
    Fill("Codec", Begin+Offset+8, 4);
    Fill("FrameRate", (float)10000000/(float)LittleEndian2int64u(Begin+Offset+16), 3);
    Fill("Width", LittleEndian2int32u(Begin+Offset+44));
    Fill("Height", LittleEndian2int32u(Begin+Offset+48));
}

//---------------------------------------------------------------------------
// Identification chunk, audio, at least 52 bytes
// "audio\0\0\0"                    8 bytes, Pos=0
// fccHandler                       4 bytes, Pos=8
// SizeOfStructure                  4 bytes, Pos=12
// TimeUnit                         8 bytes, Pos=16
// SamplesPerUnit                   8 bytes, Pos=24
// DefaultLengh                     4 bytes, Pos=32 (in media time)
// BufferSize                       4 bytes, Pos=36
// BitsPerSample                    2 bytes, Pos=40
// Reserved                         2 bytes, Pos=42
// Channels                         2 bytes, Pos=44
// BlockAlign                       2 bytes, Pos=46
// AvgBytesPerSec                   4 bytes, Pos=48
//
void File_Ogg::Identification_Audio_Analyse(size_t Offset, size_t Size)
{
    //Integrity
    if (Size<52)
        return;

    Stream_Prepare(Stream_Audio);
    Fill("Codec", Begin+Offset+8, 4);
    if (LittleEndian2int32u(Begin+Offset+48)<0x80000000) //This is a signed value, and negative values are not OK
        Fill("BitRate", LittleEndian2int32u(Begin+Offset+48)*8);
    Fill("Channel(s)", LittleEndian2int8u(Begin+Offset+44)==5?6:LittleEndian2int8u(Begin+Offset+44)); //5 channels are 5.1
    Fill("SamplingRate", LittleEndian2int64u(Begin+Offset+24));
}

//---------------------------------------------------------------------------
// Identification chunk, text
// "text\0\0\0\0"                   8 bytes, Pos=0
//
void File_Ogg::Identification_Text_Analyse(size_t Offset, size_t Size)
{
    Stream_Prepare(Stream_Text);
    Fill("Codec", "Subrip");
}

//---------------------------------------------------------------------------
// Comment chunk
// Type                             1 byte
// "vorbis"                         6 bytes
// vendor_length                    4 bytes
// vendor_string                    vendor_length bytes
// user_comment_list_HowMany        4 bytes
// length                           4 bytes
// comment                          length bytes
// length                           4 bytes
// comment                          length bytes
// (...)
// Framing bit                      1 bit
//
void File_Ogg::Comment_Analyse()
{
    if (ChunkHeader_ID.empty() || Size<12)
        return;

    if (!(CC1(Begin+Offset)==0x03 && CC6(Begin+Offset+1)==CC6("vorbis")) && !(CC1(Begin+Offset)==0x81 && CC6(Begin+Offset+1)==CC6("theora")))
        return; //Not realy a comment

    //We have comment chunk, we can start to analyse, starting to find the corresponding Identification chunk
    Identification_Analyse(ChunkHeader_ID[ChunkHeader_ID.size()-1]);

    //We can start to analyse comments
    size_t Comment_Offset=Offset+11;
    size_t Comment_Size=LittleEndian2int32u(Begin+Offset+7);
    Fill("Encoded_Library", Begin+Comment_Offset, Comment_Size, true);

    Comment_Offset+=Comment_Size;
    size_t CommentCount=LittleEndian2int32u(Begin+Comment_Offset); //Count of comments
    Comment_Offset+=4;
    while (CommentCount>0 && Comment_Offset+5<Offset+Size)
    {
        Comment_Size=LittleEndian2int32u(Begin+Comment_Offset); //Read size
        Comment_Offset+=4; //Size
        if (Comment_Offset+Comment_Size<Offset+Size)
        {
            Ztring String; String.From_UTF8((char*)(Begin+Comment_Offset), Comment_Size);
            Ztring ID=String.SubString(_T(""), _T("="));
            Ztring Text;
            if (!ID.empty())
                Text=String.SubString(_T("="), _T(""));
            else //Out of specifications!!! Grrr Doom9 :)
            {
                ID=_T("ENCODER");
                Text=String;
            }

            //Fill depends of ID
            if (0) ;
            else if (ID==_T("TITLE"))           Fill(Stream_General, 0, "Title", Text);
            else if (ID==_T("VERSION"))         Fill(Stream_General, 0, "Title/More", Text);
            else if (ID==_T("ALBUM"))           Fill(Stream_General, 0, "Album", Text);
            else if (ID==_T("TRACKNUMBER"))     Fill(Stream_General, 0, "Track", Text);
            else if (ID==_T("ARTIST"))          Fill(Stream_General, 0, "Performer", Text);
            else if (ID==_T("PERFORMER"))       Fill(Stream_General, 0, "Performer", Text);
            else if (ID==_T("COPYRIGHT"))       Fill(Stream_General, 0, "Copyright", Text);
            else if (ID==_T("LICENCE"))         Fill(Stream_General, 0, "TermsOfUse", Text);
            else if (ID==_T("ORGANIZATION"))    Fill(Stream_General, 0, "Producer", Text);
            else if (ID==_T("DESCRIPTION"))     Fill(Stream_General, 0, "Comment", Text);
            else if (ID==_T("GENRE"))           Fill(Stream_General, 0, "Genre", Text);
            else if (ID==_T("DATE"))            Fill(Stream_General, 0, "Recorded_Date", Text);
            else if (ID==_T("LOCATION"))        Fill(Stream_General, 0, "Recorded_Location", Text);
            else if (ID==_T("CONTACT"))         Fill(Stream_General, 0, "Publisher", Text);
            else if (ID==_T("ISRC"))            Fill(Stream_General, 0, "ISRC", Text);
            else if (ID==_T("COMMENT"))         Fill(Stream_General, 0, "Comment", Text);
            else if (ID==_T("COMMENTS"))        Fill(Stream_General, 0, "Comment", Text);
            else if (ID==_T("AUTHOR"))          Fill(Stream_General, 0, "Performer", Text);
            else if (ID==_T("ENCODER"))         Fill(Stream_General, 0, "Encoded_Application", Text);
            else if (ID==_T("ENCODED_USING"))   Fill(Stream_General, 0, "Encoded_Application", Text);
            else if (ID==_T("ENCODER_URL"))     Fill(Stream_General, 0, "Encoded_Application/Url", Text);
            else if (ID==_T("LWING_GAIN"))      Fill(                   "ReplayGain_Gain", Text);
            else if (ID.find(_T("CHAPTER"))==0)
            {
                if (ID.find(_T("NAME"))==Error)
                {
                    size_t Pos=Chapters_ID.size();
                    Chapters_ID.resize(Pos+1);
                    Chapters_ID[Pos]=ID.SubString(_T("CHAPTER"), _T(""));
                    Chapters_Text.resize(Pos+1);
                    Chapters_Text[Pos]=Text;
                }
                else
                {
                    Ztring ID_ToFind=ID.SubString(_T("CHAPTER"), _T("NAME"));
                    size_t Pos=0;
                    while (Pos<Chapters_ID.size() && Chapters_ID[Pos]!=ID_ToFind)
                        Pos++;
                    if (Pos<Chapters_ID.size() && Chapters_Text.size()==Chapters_ID.size())
                    {
                        Chapters_Text[Pos]+=_T(" ");
                        Chapters_Text[Pos]+=Text;
                    }
                }
            }
            else                                Fill(                   ID.To_Local().c_str(), Text);
            /*
            else if (ID==_T("LANGUAGE") && Commentaire[Commentaire_EnCours]=='A') {Audio[Commentaire_I[Commentaire_EnCours]](_T("Language/String"))=Texte; Audio[Commentaire_I[Commentaire_EnCours]](_T("Language"))=Ztring(Texte.c_str(),2);}
            else if (ID==_T("LANGUAGE") && Commentaire[Commentaire_EnCours]=='T') {Text[Commentaire_I[Commentaire_EnCours]](_T("Language/String"))=Texte; Text[Commentaire_I[Commentaire_EnCours]](_T("Language"))=Ztring(Texte.c_str(),2);}
            else if (Commentaire[Commentaire_EnCours]=='A')
            {
                //TODO : Chaine_Tableau2 ou 1 gere mal quand Chaine[0] est vide et pas Chaine[1]
                //TODO : RechercherEntre : si pas _T("="), retorune Begin, pas cool
                if (ID==_T(""))
                {
                    ZtringList C3=Ztring(Ztring(_T("(Note by MediaInfo, Out of specifications);"))+Texte);
                    C3(Info_Name_Text)=_T("(Note by MediaInfo, Out of specifications)");
                    Audio[Commentaire_I[Commentaire_EnCours]].push_back(C3);
                }
                else
                {
                    ZtringList C3=Ztring(C2.SubString(_T(""), _T("="))+_T(";")+C2.SubString(_T("="), _T("")));
                    Ztring C4=C3(0);
                    C3(Info_Name_Text)=C4;
                    Audio[Commentaire_I[Commentaire_EnCours]].push_back(C3);
                }
            }
            else if (Commentaire[Commentaire_EnCours]=='T') Text[Commentaire_I[Commentaire_EnCours]].push_back((C2.SubString(_T(""), _T("="))+_T(";")+C2.SubString(_T("="), _T(""))).c_str()); //Visual C++ patch, doesn't like push_back overload
            else
            {
                 General[0].push_back(C2.SubString(_T(""), _T("=")));
                 General[0](C2.SubString(_T(""), _T("=")), Info_Name_Text)=C2.SubString(_T(""), _T("="));
                 General[0](C2.SubString(_T(""), _T("=")))=C2.SubString(_T("="), _T(""));
            }
            */

            CommentCount--;
            Comment_Offset+=Comment_Size;
        }
        else
            CommentCount=0;
    }

    if (!Chapters_ID.empty() && Chapters_Text.size()==Chapters_ID.size())
    {
        Stream_Prepare(Stream_Chapters);
        for (size_t Pos=0; Pos<Chapters_ID.size(); Pos++)
            Fill(Chapters_ID[Pos].To_Local().c_str(), Chapters_Text[Pos]);
        Fill("Total", Chapters_ID[Chapters_ID.size()-1].To_int32u());
        Chapters_ID.clear();
        Chapters_Text.clear();
    }

}

//---------------------------------------------------------------------------
// Seutup chunk
//
void File_Ogg::Setup_Analyse()
{
    ShouldStop=true; //At this step, header information is over
}

//---------------------------------------------------------------------------
void File_Ogg::HowTo(stream_t StreamKind)
{
         if (StreamKind==Stream_General)
    {
        General[0](_T("Format"), Info_HowTo)=_T("R");
        General[0](_T("OveralBitRate"), Info_HowTo)=_T("R");
        General[0](_T("PlayTime"), Info_HowTo)=_T("R");
        General[0](_T("Album"), Info_HowTo)=_T("R ALBUM");
        General[0](_T("Movie"), Info_HowTo)=_T("R TITLE");
        General[0](_T("Movie/More"), Info_HowTo)=_T("R VERSION");
        General[0](_T("Track"), Info_HowTo)=_T("R TITLE");
        General[0](_T("Track/More"), Info_HowTo)=_T("R VERSION");
        General[0](_T("Author"), Info_HowTo)=_T("R ARTIST");
        General[0](_T("Encoded_Date"), Info_HowTo)=_T("R DATE");
        General[0](_T("Recorded_Location"), Info_HowTo)=_T("R LOCATION");
        General[0](_T("Copyright"), Info_HowTo)=_T("R COPYRIGHT");
        General[0](_T("Comment"), Info_HowTo)=_T("R DESCRIPTION or COMMENT");
    }
    else if (StreamKind==Stream_Video)
    {
        Video[0](_T("Codec"), Info_HowTo)=_T("R");
        Video[0](_T("BitRate"), Info_HowTo)=_T("R");
        Video[0](_T("Width"), Info_HowTo)=_T("R");
        Video[0](_T("Height"), Info_HowTo)=_T("R");
        Video[0](_T("AspectRatio"), Info_HowTo)=_T("R");
        Video[0](_T("FrameRate"), Info_HowTo)=_T("R");
    }
    else if (StreamKind==Stream_Audio)
    {
        Audio[0](_T("Codec"), Info_HowTo)=_T("R");
        Audio[0](_T("BitRate"), Info_HowTo)=_T("R");
        Audio[0](_T("Channel(s)"), Info_HowTo)=_T("R");
        Audio[0](_T("SamplingRate"), Info_HowTo)=_T("R");
        Audio[0](_T("Language"), Info_HowTo)=_T("R LANGUAGE");
        Audio[0](_T("Language/String"), Info_HowTo)=_T("R");
        Audio[0](_T("Language/Info"), Info_HowTo)=_T("R");
    }
    else if (StreamKind==Stream_Text)
    {
        Text[0](_T("Codec"), Info_HowTo)=_T("R");
        Text[0](_T("Language"), Info_HowTo)=_T("R");
        Text[0](_T("Language/String"), Info_HowTo)=_T("R LANGUAGE");
        Text[0](_T("Language/Info"), Info_HowTo)=_T("R");
    }
    else if (StreamKind==Stream_Chapters)
    {
        Chapters[0](_T("Total"), Info_HowTo)=_T("R");
    }
}

} //NameSpace

#endif //MEDIAINFO_OGG_*
